<?php
class Notification {
    private $settings;
    
    public function __construct() {
        $config_file = __DIR__ . '/../config/notification_settings.php';
        if (file_exists($config_file)) {
            $this->settings = require $config_file;
        } else {
            $this->settings = [
                'enable_email' => 0,
                'enable_wechat' => 0
            ];
        }
    }
    
    /**
     * 发送新举报通知
     * 
     * @param array $report 举报信息
     * @return bool 是否成功发送
     */
    public function sendNewReportNotification($report) {
        $success = true;
        
        if ($this->settings['enable_email'] == 1) {
            $success = $success && $this->sendEmailNotification($report);
        }
        
        if ($this->settings['enable_wechat'] == 1) {
            $success = $success && $this->sendWechatNotification($report);
        }
        
        return $success;
    }
    
    /**
     * 发送邮件通知
     * 
     * @param array $report 举报信息
     * @return bool 是否成功发送
     */
    private function sendEmailNotification($report) {
        // 检查必要的设置
        if (empty($this->settings['smtp_server']) || 
            empty($this->settings['smtp_username']) || 
            empty($this->settings['email_recipients'])) {
            return false;
        }
        
        // 引入PHPMailer
        require_once __DIR__ . '/../vendor/autoload.php';
        
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        
        try {
            // 服务器设置
            $mail->isSMTP();
            $mail->Host = $this->settings['smtp_server'];
            $mail->SMTPAuth = true;
            $mail->Username = $this->settings['smtp_username'];
            $mail->Password = $this->settings['smtp_password'];
            $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
            $mail->Port = $this->settings['smtp_port'];
            $mail->CharSet = 'UTF-8';
            
            // 收件人
            $recipients = explode(',', $this->settings['email_recipients']);
            foreach ($recipients as $recipient) {
                $mail->addAddress(trim($recipient));
            }
            
            // 发件人
            $mail->setFrom($this->settings['email_sender'], '举报系统');
            
            // 内容
            $mail->isHTML(true);
            $mail->Subject = '新举报提醒 - ' . $report['report_type'];
            
            // 构建邮件内容
            $content = "<h2>收到新的举报</h2>";
            $content .= "<p><strong>举报类型：</strong>" . htmlspecialchars($report['report_type']) . "</p>";
            $content .= "<p><strong>举报人：</strong>" . htmlspecialchars($report['reporter_name']) . "</p>";
            $content .= "<p><strong>联系方式：</strong>" . htmlspecialchars($report['reporter_contact']) . "</p>";
            $content .= "<p><strong>举报内容：</strong></p>";
            $content .= "<div style='background-color: #f5f5f5; padding: 10px; border-radius: 5px;'>";
            $content .= nl2br(htmlspecialchars($report['report_content']));
            $content .= "</div>";
            
            if (!empty($report['reported_url'])) {
                $content .= "<p><strong>相关链接：</strong><a href='" . htmlspecialchars($report['reported_url']) . "'>" . htmlspecialchars($report['reported_url']) . "</a></p>";
            }
            
            $content .= "<p><a href='http://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . "/../admin/view_report.php?id=" . $report['id'] . "' style='display: inline-block; background-color: #0d6efd; color: white; padding: 10px 15px; text-decoration: none; border-radius: 5px;'>查看详情</a></p>";
            
            $mail->Body = $content;
            
            $mail->send();
            return true;
        } catch (Exception $e) {
            error_log('邮件发送失败: ' . $mail->ErrorInfo);
            return false;
        }
    }
    
    /**
     * 发送企业微信通知
     * 
     * @param array $report 举报信息
     * @return bool 是否成功发送
     */
    private function sendWechatNotification($report) {
        // 检查必要的设置
        if (empty($this->settings['webhook_url'])) {
            error_log('企业微信Webhook URL未设置');
            return false;
        }
        
        // 构建消息内容
        $message = [
            'msgtype' => 'markdown',
            'markdown' => [
                'content' => "### 收到新的举报\n" .
                    "> **举报类型**：" . $report['report_type'] . "\n" .
                    "> **举报人**：" . $report['reporter_name'] . "\n" .
                    "> **联系方式**：" . $report['reporter_contact'] . "\n" .
                    "> **举报内容**：" . mb_substr($report['report_content'], 0, 100) . (mb_strlen($report['report_content']) > 100 ? '...' : '') . "\n" .
                    "> **提交时间**：" . date('Y-m-d H:i:s', strtotime($report['created_at'])) . "\n"
            ]
        ];
        
        // 添加详情链接
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";
        $domain = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $path = dirname(dirname($_SERVER['PHP_SELF'] ?? ''));
        $adminUrl = $protocol . $domain . $path . "/admin/view_report.php?id=" . $report['id'];
        
        $message['markdown']['content'] .= "\n[点击查看详情](" . $adminUrl . ")";
        
        // 记录调试信息
        error_log('准备发送企业微信通知: ' . json_encode($message, JSON_UNESCAPED_UNICODE));
        
        // 发送请求
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->settings['webhook_url']);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // 增加连接超时时间
        curl_setopt($ch, CURLOPT_TIMEOUT, 15); // 增加请求超时时间
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json;charset=utf-8']);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message, JSON_UNESCAPED_UNICODE));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        
        // 记录详细调试信息
        error_log('企业微信通知响应: ' . $response . ', HTTP状态码: ' . $httpCode);
        if (!empty($curlError)) {
            error_log('企业微信通知CURL错误: ' . $curlError);
        }
        
        curl_close($ch);
        
        if ($httpCode == 200) {
            $result = json_decode($response, true);
            if (isset($result['errcode']) && $result['errcode'] == 0) {
                error_log('企业微信通知发送成功');
                return true;
            } else {
                error_log('企业微信通知发送失败，错误码: ' . ($result['errcode'] ?? '未知') . ', 错误信息: ' . ($result['errmsg'] ?? '未知'));
                return false;
            }
        } else {
            error_log('企业微信通知HTTP请求失败，状态码: ' . $httpCode);
            return false;
        }
    }
    
    /**
     * 发送测试企业微信通知
     * 
     * @param string $webhook_url 可选的Webhook URL
     * @return array 发送结果
     */
    public function sendTestWechatNotification($webhook_url = null) {
        // 使用提供的webhook_url或设置中的webhook_url
        $url = $webhook_url ?: $this->settings['webhook_url'];
        
        // 检查必要的设置
        if (empty($url)) {
            return [
                'success' => false,
                'message' => '企业微信Webhook URL未设置'
            ];
        }
        
        // 验证URL格式
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            return [
                'success' => false,
                'message' => '无效的Webhook URL格式'
            ];
        }
        
        // 确保URL以http://或https://开头
        if (!preg_match('/^https?:\/\//i', $url)) {
            return [
                'success' => false,
                'message' => '无效的URL格式，必须以http://或https://开头'
            ];
        }
        
        // 构建测试消息
        $message = [
            'msgtype' => 'markdown',
            'markdown' => [
                'content' => "### 举报系统 - 测试消息\n" .
                    "> **测试时间**：" . date('Y-m-d H:i:s') . "\n" .
                    "> **系统状态**：正常\n" .
                    "\n这是一条测试消息，如果您收到此消息，说明举报系统的企业微信通知功能已配置成功。"
            ]
        ];
        
        // 记录调试信息
        error_log('准备发送企业微信测试通知: ' . json_encode($message, JSON_UNESCAPED_UNICODE));
        error_log('企业微信Webhook URL: ' . $url);
        
        // 发送请求
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json;charset=utf-8']);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message, JSON_UNESCAPED_UNICODE));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        
        // 执行请求并获取响应
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
        
        // 记录详细调试信息
        error_log('企业微信测试通知响应: ' . $response);
        error_log('HTTP状态码: ' . $httpCode);
        error_log('Content-Type: ' . $contentType);
        if (!empty($curlError)) {
            error_log('企业微信测试通知CURL错误: ' . $curlError);
        }
        
        curl_close($ch);
        
        // 检查是否为HTML响应（通常表示出错）
        $isHtmlResponse = (strpos($contentType, 'text/html') !== false) || 
                          (strpos($response, '<!DOCTYPE') !== false) || 
                          (strpos($response, '<html') !== false);
        
        if ($isHtmlResponse) {
            // 提取HTML中的错误信息
            $errorMessage = '服务器返回了HTML而不是JSON';
            if (preg_match('/<title>(.*?)<\/title>/i', $response, $matches)) {
                $errorMessage .= ' - ' . $matches[1];
            }
            
            // 截取部分HTML内容作为错误提示
            $shortResponse = mb_substr(strip_tags($response), 0, 200) . '...';
            
            return [
                'success' => false,
                'message' => '企业微信通知失败: ' . $errorMessage . "\n\n响应内容: " . $shortResponse,
                'http_code' => $httpCode
            ];
        }
        
        if ($httpCode == 200) {
            // 尝试解析JSON响应
            $result = json_decode($response, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                if (isset($result['errcode']) && $result['errcode'] == 0) {
                    return [
                        'success' => true,
                        'message' => '企业微信测试通知发送成功！'
                    ];
                } else {
                    return [
                        'success' => false,
                        'message' => '企业微信通知发送失败，错误码: ' . ($result['errcode'] ?? '未知') . ', 错误信息: ' . ($result['errmsg'] ?? '未知')
                    ];
                }
            } else {
                // JSON解析失败
                return [
                    'success' => false,
                    'message' => '企业微信返回了无效的JSON响应: ' . mb_substr($response, 0, 200) . '...',
                    'json_error' => json_last_error_msg()
                ];
            }
        } else {
            return [
                'success' => false,
                'message' => '企业微信通知HTTP请求失败，状态码: ' . $httpCode . (empty($curlError) ? '' : ', CURL错误: ' . $curlError)
            ];
        }
    }
}