<?php
session_start();

// 检查用户是否已登录
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php?error=请先登录");
    exit;
}

require_once 'db_connect.php';

// 确保数据库连接变量名正确
if (!isset($pdo) && isset($conn)) {
    $pdo = $conn;
}

$user_id = $_SESSION['user_id'];
$success_message = '';
$error_message = '';

// 处理表单提交
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $current_password = $_POST['current_password'];
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];
    
    // 验证输入
    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
        $error_message = "请填写所有密码字段";
    } elseif (strlen($new_password) < 6) {
        $error_message = "新密码长度至少为6个字符";
    } elseif ($new_password !== $confirm_password) {
        $error_message = "两次输入的新密码不一致";
    } else {
        try {
            // 获取用户当前密码哈希
            if ($pdo instanceof PDO) {
                $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
            } else {
                $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
                $stmt->bind_param("i", $user_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $user = $result->fetch_assoc();
            }
            
            if (!$user) {
                $error_message = "找不到用户信息";
            } elseif (!password_verify($current_password, $user['password'])) {
                $error_message = "当前密码不正确";
            } else {
                // 更新密码
                $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                
                if ($pdo instanceof PDO) {
                    $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
                    $stmt->execute([$hashed_password, $user_id]);
                } else {
                    $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
                    $stmt->bind_param("si", $hashed_password, $user_id);
                    $stmt->execute();
                }
                
                $success_message = "密码修改成功";
            }
        } catch (Exception $e) {
            $error_message = "修改密码失败: " . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>修改密码 - 举报系统</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            padding-top: 20px;
            padding-bottom: 20px;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            background-attachment: fixed;
            position: relative;
        }
        
        body::before {
            content: "";
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-image: url('https://cdn.pixabay.com/photo/2018/03/22/02/37/background-3249063_1280.png');
            background-size: cover;
            background-position: center;
            opacity: 0.1;
            z-index: -1;
        }
        
        .container {
            max-width: 700px;
            position: relative;
            z-index: 1;
        }
        
        .card {
            border-radius: 10px;
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
            border: none;
            background-color: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(5px);
        }
        
        .card-body {
            padding: 30px;
        }
        
        h1 {
            color: #0d6efd;
            font-weight: 600;
            text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.1);
        }
        
        .btn-back {
            color: #6c757d;
            text-decoration: none;
            transition: color 0.3s;
            display: inline-flex;
            align-items: center;
        }
        
        .btn-back:hover {
            color: #0d6efd;
        }
        
        .form-label {
            font-weight: 500;
        }
        
        .form-control:focus {
            border-color: #86b7fe;
            box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
        }
        
        .password-toggle {
            cursor: pointer;
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            z-index: 10;
            color: #6c757d;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1>修改密码</h1>
            <a href="user_dashboard.php" class="btn-back">
                <i class="bi bi-arrow-left me-2"></i> 返回用户平台
            </a>
        </div>
        
        <?php if (!empty($success_message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="bi bi-check-circle-fill me-2"></i>
            <?php echo htmlspecialchars($success_message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($error_message)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="bi bi-exclamation-triangle-fill me-2"></i>
            <?php echo htmlspecialchars($error_message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-body">
                <div class="mb-4 text-center">
                    <div class="display-1 text-primary">
                        <i class="bi bi-key"></i>
                    </div>
                    <h5>修改您的账户密码</h5>
                </div>
                
                <form method="post" action="change_password.php">
                    <div class="mb-3 position-relative">
                        <label for="current_password" class="form-label">当前密码</label>
                        <input type="password" class="form-control" id="current_password" name="current_password" required>
                        <i class="bi bi-eye-slash password-toggle" data-target="current_password"></i>
                    </div>
                    
                    <div class="mb-3 position-relative">
                        <label for="new_password" class="form-label">新密码</label>
                        <input type="password" class="form-control" id="new_password" name="new_password" required>
                        <i class="bi bi-eye-slash password-toggle" data-target="new_password"></i>
                        <div class="form-text">密码长度至少为6个字符</div>
                    </div>
                    
                    <div class="mb-4 position-relative">
                        <label for="confirm_password" class="form-label">确认新密码</label>
                        <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                        <i class="bi bi-eye-slash password-toggle" data-target="confirm_password"></i>
                    </div>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-lg me-2"></i>修改密码
                        </button>
                        <a href="user_dashboard.php" class="btn btn-outline-secondary">
                            <i class="bi bi-x-lg me-2"></i>取消
                        </a>
                    </div>
                </form>
            </div>
        </div>
        
        <footer class="mt-5 text-center text-muted">
            <small>&copy; <?php echo date('Y'); ?> 举报系统 - 共建健康网络环境</small>
        </footer>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // 密码显示/隐藏切换
        const toggles = document.querySelectorAll('.password-toggle');
        toggles.forEach(toggle => {
            toggle.addEventListener('click', function() {
                const targetId = this.getAttribute('data-target');
                const passwordInput = document.getElementById(targetId);
                
                if (passwordInput.type === 'password') {
                    passwordInput.type = 'text';
                    this.classList.remove('bi-eye-slash');
                    this.classList.add('bi-eye');
                } else {
                    passwordInput.type = 'password';
                    this.classList.remove('bi-eye');
                    this.classList.add('bi-eye-slash');
                }
            });
        });
    });
    </script>
</body>
</html>