import React from 'react';
import { motion } from 'framer-motion';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Switch } from './ui/switch';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Slider } from './ui/slider';
import { Label } from './ui/label';
import { Separator } from './ui/separator';
import { useToast } from './ui/use-toast';
import { 
  Volume2, 
  Zap, 
  Palette, 
  Globe, 
  Download,
  Upload,
  Trash2,
  RefreshCw
} from 'lucide-react';
import { Student, PickResult, Settings } from '../App';

interface SettingsPanelProps {
  settings: Settings;
  onSettingsChange: (settings: Settings) => void;
  students: Student[];
  history: PickResult[];
}

export function SettingsPanel({ 
  settings, 
  onSettingsChange, 
  students, 
  history 
}: SettingsPanelProps) {
  const { toast } = useToast();

  const handleSettingChange = (key: keyof Settings, value: any) => {
    const newSettings = { ...settings, [key]: value };
    onSettingsChange(newSettings);
    
    toast({
      title: "设置已更新",
      description: "您的设置已保存",
    });
  };

  const exportData = () => {
    const data = {
      students,
      history,
      settings,
      exportTime: new Date().toISOString()
    };
    
    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `柯猿课堂互动系统数据_${new Date().toLocaleDateString('zh-CN')}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    toast({
      title: "导出成功",
      description: "数据已导出到本地文件",
    });
  };

  const importData = () => {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = '.json';
    input.onchange = (e) => {
      const file = (e.target as HTMLInputElement).files?.[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = (e) => {
          try {
            const data = JSON.parse(e.target?.result as string);
            if (data.settings) {
              onSettingsChange(data.settings);
              toast({
                title: "导入成功",
                description: "设置已从文件中恢复",
              });
            }
          } catch (error) {
            toast({
              title: "导入失败",
              description: "文件格式不正确",
              variant: "destructive",
            });
          }
        };
        reader.readAsText(file);
      }
    };
    input.click();
  };

  const clearAllData = () => {
    if (confirm('确定要清除所有数据吗？此操作不可恢复！')) {
      localStorage.clear();
      window.location.reload();
    }
  };

  const resetSettings = () => {
    const defaultSettings: Settings = {
      soundEnabled: true,
      soundType: 'bell',
      volume: 0.7,
      animationEnabled: true,
      animationSpeed: 'normal',
      animationType: 'fade',
      theme: 'light',
      language: 'zh-CN'
    };
    
    onSettingsChange(defaultSettings);
    
    toast({
      title: "设置已重置",
      description: "所有设置已恢复为默认值",
    });
  };

  return (
    <div className="space-y-6">
      {/* 页面标题 */}
      <motion.div
        initial={{ opacity: 0, y: -20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5 }}
      >
        <h1 className="text-3xl font-bold text-gray-900 mb-2">系统设置</h1>
        <p className="text-gray-600">个性化配置和偏好设置</p>
      </motion.div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* 声音设置 */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.1 }}
        >
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Volume2 className="w-5 h-5 text-blue-600" />
                声音设置
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <Label htmlFor="sound-enabled">启用声音效果</Label>
                <Switch
                  id="sound-enabled"
                  checked={settings.soundEnabled}
                  onCheckedChange={(checked) => handleSettingChange('soundEnabled', checked)}
                />
              </div>

              <div className="space-y-2">
                <Label>声音类型</Label>
                <Select
                  value={settings.soundType}
                  onValueChange={(value) => handleSettingChange('soundType', value)}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="bell">铃声</SelectItem>
                    <SelectItem value="applause">掌声</SelectItem>
                    <SelectItem value="music">音乐</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>音量 ({Math.round(settings.volume * 100)}%)</Label>
                <Slider
                  value={[settings.volume]}
                  onValueChange={([value]) => handleSettingChange('volume', value)}
                  max={1}
                  min={0}
                  step={0.1}
                  className="w-full"
                />
              </div>
            </CardContent>
          </Card>
        </motion.div>

        {/* 动画设置 */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Zap className="w-5 h-5 text-purple-600" />
                动画设置
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <Label htmlFor="animation-enabled">启用动画效果</Label>
                <Switch
                  id="animation-enabled"
                  checked={settings.animationEnabled}
                  onCheckedChange={(checked) => handleSettingChange('animationEnabled', checked)}
                />
              </div>

              <div className="space-y-2">
                <Label>动画速度</Label>
                <Select
                  value={settings.animationSpeed}
                  onValueChange={(value) => handleSettingChange('animationSpeed', value)}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="slow">慢速</SelectItem>
                    <SelectItem value="normal">正常</SelectItem>
                    <SelectItem value="fast">快速</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>动画类型</Label>
                <Select
                  value={settings.animationType}
                  onValueChange={(value) => handleSettingChange('animationType', value)}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="fade">淡入淡出</SelectItem>
                    <SelectItem value="rotate">旋转</SelectItem>
                    <SelectItem value="scale">缩放</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>
        </motion.div>

        {/* 外观设置 */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Palette className="w-5 h-5 text-pink-600" />
                外观设置
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label>主题模式</Label>
                <Select
                  value={settings.theme}
                  onValueChange={(value) => handleSettingChange('theme', value)}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="light">浅色模式</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>语言设置</Label>
                <Select
                  value={settings.language}
                  onValueChange={(value) => handleSettingChange('language', value)}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="zh-CN">简体中文</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>
        </motion.div>

        {/* 数据管理 */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.4 }}
        >
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Globe className="w-5 h-5 text-green-600" />
                数据管理
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-3">
                <Button
                  onClick={exportData}
                  variant="outline"
                  className="w-full"
                >
                  <Download className="w-4 h-4 mr-2" />
                  导出备份
                </Button>
                
                <Button
                  onClick={importData}
                  variant="outline"
                  className="w-full"
                >
                  <Upload className="w-4 h-4 mr-2" />
                  导入备份
                </Button>
              </div>

              <Separator />

              <div className="space-y-3">
                <Button
                  onClick={resetSettings}
                  variant="outline"
                  className="w-full"
                >
                  <RefreshCw className="w-4 h-4 mr-2" />
                  重置设置
                </Button>

                <Button
                  onClick={clearAllData}
                  variant="destructive"
                  className="w-full"
                >
                  <Trash2 className="w-4 h-4 mr-2" />
                  清除所有数据
                </Button>
              </div>
            </CardContent>
          </Card>
        </motion.div>
      </div>

      {/* 系统信息 */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5, delay: 0.5 }}
      >
        <Card>
          <CardHeader>
            <CardTitle>系统信息</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
              <div>
                <div className="font-medium text-gray-700">学生数量</div>
                <div className="text-2xl font-bold text-blue-600">{students.length}</div>
              </div>
              <div>
                <div className="font-medium text-gray-700">历史记录</div>
                <div className="text-2xl font-bold text-green-600">{history.length}</div>
              </div>
              <div>
                <div className="font-medium text-gray-700">版本信息</div>
                <div className="text-lg font-semibold text-purple-600">v0.0.2</div>
              </div>
              <div>
                <div className="font-medium text-gray-700">最后更新</div>
                <div className="text-lg font-semibold text-orange-600">2025</div>
              </div>
            </div>
          </CardContent>
        </Card>
      </motion.div>
    </div>
  );
}