import React from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { ScrollArea } from './ui/scroll-area';
import { Trophy, Clock, Users, History } from 'lucide-react';
import type { PickResult, Settings } from '../App';

interface ResultDisplayProps {
  currentResult: PickResult | null;
  history: PickResult[];
  settings: Settings;
}

export function ResultDisplay({ currentResult, history, settings }: ResultDisplayProps) {
  const formatTime = (date: Date) => {
    return date.toLocaleTimeString('zh-CN', {
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit'
    });
  };

  const getModeText = (mode: string) => {
    switch (mode) {
      case 'single': return '单人';
      case 'multiple': return '多人';
      case 'group': return '分组';
      default: return mode;
    }
  };

  const getAnimationVariants = () => {
    switch (settings.animationType) {
      case 'rotate':
        return {
          initial: { opacity: 0, rotate: -180, scale: 0.5 },
          animate: { opacity: 1, rotate: 0, scale: 1 },
          exit: { opacity: 0, rotate: 180, scale: 0.5 }
        };
      case 'scale':
        return {
          initial: { opacity: 0, scale: 0 },
          animate: { opacity: 1, scale: 1 },
          exit: { opacity: 0, scale: 0 }
        };
      default: // fade
        return {
          initial: { opacity: 0, y: 20 },
          animate: { opacity: 1, y: 0 },
          exit: { opacity: 0, y: -20 }
        };
    }
  };

  const getAnimationDuration = () => {
    switch (settings.animationSpeed) {
      case 'slow': return 0.8;
      case 'fast': return 0.3;
      default: return 0.5;
    }
  };

  return (
    <Card className="h-full">
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <Trophy className="h-5 w-5" />
          <span>点名结果</span>
        </CardTitle>
      </CardHeader>
      
      <CardContent className="space-y-6">
        {/* 当前结果展示 */}
        <div className="space-y-3">
          <h3 className="text-sm font-medium text-gray-700">当前结果</h3>
          
          <div className="min-h-32 flex items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-50 rounded-lg p-4">
            <AnimatePresence mode="wait">
              {currentResult ? (
                <motion.div
                  key={currentResult.timestamp.getTime()}
                  variants={getAnimationVariants()}
                  initial="initial"
                  animate="animate"
                  exit="exit"
                  transition={{ 
                    duration: settings.animationEnabled ? getAnimationDuration() : 0.2,
                    ease: "easeInOut"
                  }}
                  className="text-center space-y-3"
                >
                  <div className="flex flex-wrap justify-center gap-2">
                    {currentResult.students.map((student, index) => (
                      <motion.div
                        key={student.id}
                        initial={{ opacity: 0, scale: 0.8 }}
                        animate={{ opacity: 1, scale: 1 }}
                        transition={{ 
                          delay: index * 0.1,
                          duration: settings.animationEnabled ? getAnimationDuration() : 0.2
                        }}
                      >
                        <Badge 
                          variant="default" 
                          className="text-lg px-4 py-2 bg-gradient-to-r from-blue-500 to-indigo-600"
                        >
                          {student.name}
                        </Badge>
                      </motion.div>
                    ))}
                  </div>
                  
                  <div className="flex items-center justify-center space-x-4 text-sm text-gray-600">
                    <div className="flex items-center space-x-1">
                      <Users className="h-4 w-4" />
                      <span>{getModeText(currentResult.mode)}</span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <Clock className="h-4 w-4" />
                      <span>{formatTime(currentResult.timestamp)}</span>
                    </div>
                  </div>
                </motion.div>
              ) : (
                <motion.div
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  className="text-center text-gray-500"
                >
                  <Trophy className="h-12 w-12 mx-auto mb-2 text-gray-300" />
                  <p className="text-sm">点击开始点名</p>
                  <p className="text-xs">结果将在这里显示</p>
                </motion.div>
              )}
            </AnimatePresence>
          </div>
        </div>

        {/* 历史记录 */}
        <div className="space-y-3">
          <div className="flex items-center justify-between">
            <h3 className="text-sm font-medium text-gray-700">历史记录</h3>
            <Badge variant="secondary">{history.length}</Badge>
          </div>
          
          <ScrollArea className="h-64">
            <div className="space-y-2">
              <AnimatePresence>
                {history.map((record, index) => (
                  <motion.div
                    key={`${record.timestamp.getTime()}-${index}`}
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                    exit={{ opacity: 0, x: 20 }}
                    transition={{ duration: 0.2, delay: index * 0.05 }}
                    className="p-3 bg-gray-50 rounded-md space-y-2"
                  >
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-2">
                        <Badge variant="outline" className="text-xs">
                          {getModeText(record.mode)}
                        </Badge>
                        <span className="text-xs text-gray-500">
                          {formatTime(record.timestamp)}
                        </span>
                      </div>
                      <span className="text-xs text-gray-500">
                        {record.students.length}人
                      </span>
                    </div>
                    
                    <div className="flex flex-wrap gap-1">
                      {record.students.map((student) => (
                        <Badge 
                          key={student.id} 
                          variant="secondary" 
                          className="text-xs"
                        >
                          {student.name}
                        </Badge>
                      ))}
                    </div>
                  </motion.div>
                ))}
              </AnimatePresence>
              
              {history.length === 0 && (
                <div className="text-center py-8 text-gray-500">
                  <History className="h-8 w-8 mx-auto mb-2 text-gray-300" />
                  <p className="text-sm">暂无历史记录</p>
                  <p className="text-xs">开始点名后记录将显示在这里</p>
                </div>
              )}
            </div>
          </ScrollArea>
        </div>
      </CardContent>
    </Card>
  );
}