import React, { useState, useEffect } from 'react';
import { Button } from './ui/button';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Progress } from './ui/progress';
import { useToast } from './ui/use-toast';
import { 
  Vote, 
  Plus, 
  Trash2, 
  Play, 
  Square, 
  BarChart3, 
  Users, 
  Clock,
  CheckCircle,
  XCircle,
  PieChart,
  Download,
  Eye,
  EyeOff,
  Shuffle
} from 'lucide-react';
import { Student } from '../App';

interface VoteOption {
  id: string;
  text: string;
  votes: number;
  voters: string[]; // 投票者ID列表
}

interface Poll {
  id: string;
  question: string;
  description?: string;
  options: VoteOption[];
  isActive: boolean;
  isAnonymous: boolean;
  allowMultiple: boolean;
  createdAt: Date;
  endedAt?: Date;
  totalVotes: number;
}

interface ClassroomVotingProps {
  students: Student[];
}

export function ClassroomVoting({ students }: ClassroomVotingProps) {
  const { toast } = useToast();
  const [polls, setPolls] = useState<Poll[]>([]);
  const [currentPoll, setCurrentPoll] = useState<Poll | null>(null);
  const [isCreating, setIsCreating] = useState(false);
  const [showResults, setShowResults] = useState(false);
  
  // 创建投票表单状态
  const [newPoll, setNewPoll] = useState({
    question: '',
    description: '',
    options: ['', ''],
    isAnonymous: true,
    allowMultiple: false
  });

  // 从localStorage加载数据
  useEffect(() => {
    const savedPolls = localStorage.getItem('classroom-polls');
    if (savedPolls) {
      try {
        const parsedPolls = JSON.parse(savedPolls);
        const pollsWithDates = parsedPolls.map((poll: any) => ({
          ...poll,
          createdAt: new Date(poll.createdAt),
          endedAt: poll.endedAt ? new Date(poll.endedAt) : undefined
        }));
        setPolls(pollsWithDates);
        
        // 恢复活跃的投票
        const activePoll = pollsWithDates.find((poll: Poll) => poll.isActive);
        if (activePoll) {
          setCurrentPoll(activePoll);
        }
      } catch (error) {
        console.error('Failed to load polls:', error);
      }
    }
  }, []);

  // 保存数据到localStorage
  useEffect(() => {
    localStorage.setItem('classroom-polls', JSON.stringify(polls));
  }, [polls]);

  // 创建新投票
  const createPoll = () => {
    if (!newPoll.question.trim()) {
      toast({
        title: "⚠️ 请输入问题",
        description: "投票问题不能为空",
        variant: "destructive"
      });
      return;
    }

    const validOptions = newPoll.options.filter(opt => opt.trim());
    if (validOptions.length < 2) {
      toast({
        title: "⚠️ 选项不足",
        description: "至少需要2个有效选项",
        variant: "destructive"
      });
      return;
    }

    const poll: Poll = {
      id: Date.now().toString(),
      question: newPoll.question.trim(),
      description: newPoll.description.trim(),
      options: validOptions.map((text, index) => ({
        id: `option-${index}`,
        text: text.trim(),
        votes: 0,
        voters: []
      })),
      isActive: false,
      isAnonymous: newPoll.isAnonymous,
      allowMultiple: newPoll.allowMultiple,
      createdAt: new Date(),
      totalVotes: 0
    };

    setPolls(prev => [poll, ...prev]);
    setNewPoll({
      question: '',
      description: '',
      options: ['', ''],
      isAnonymous: true,
      allowMultiple: false
    });
    setIsCreating(false);

    toast({
      title: "✅ 投票创建成功",
      description: "可以开始投票了"
    });
  };

  // 开始投票
  const startPoll = (pollId: string) => {
    // 结束当前活跃的投票
    if (currentPoll) {
      endPoll(currentPoll.id);
    }

    setPolls(prev => prev.map(poll => ({
      ...poll,
      isActive: poll.id === pollId
    })));

    const poll = polls.find(p => p.id === pollId);
    if (poll) {
      setCurrentPoll({ ...poll, isActive: true });
      setShowResults(false);
      
      toast({
        title: "🗳️ 投票开始",
        description: `"${poll.question}" 投票已开始`
      });
    }
  };

  // 结束投票
  const endPoll = (pollId: string) => {
    const now = new Date();
    setPolls(prev => prev.map(poll => 
      poll.id === pollId 
        ? { ...poll, isActive: false, endedAt: now }
        : poll
    ));

    if (currentPoll && currentPoll.id === pollId) {
      setCurrentPoll(prev => prev ? { ...prev, isActive: false, endedAt: now } : null);
      setShowResults(true);
      
      toast({
        title: "⏹️ 投票结束",
        description: "投票已结束，可以查看结果"
      });
    }
  };

  // 模拟学生投票
  const simulateVote = (optionId: string) => {
    if (!currentPoll || !currentPoll.isActive) return;

    // 随机选择一个学生进行投票
    const availableStudents = students.filter(student => {
      if (currentPoll.allowMultiple) {
        return true; // 允许多选时，学生可以重复投票
      } else {
        // 单选时，检查学生是否已经投过票
        return !currentPoll.options.some(option => 
          option.voters.includes(student.id)
        );
      }
    });

    if (availableStudents.length === 0) {
      toast({
        title: "ℹ️ 无法投票",
        description: "所有学生都已投票",
        variant: "default"
      });
      return;
    }

    const randomStudent = availableStudents[Math.floor(Math.random() * availableStudents.length)];
    
    // 更新投票数据
    const updatedPoll = {
      ...currentPoll,
      options: currentPoll.options.map(option => {
        if (option.id === optionId) {
          return {
            ...option,
            votes: option.votes + 1,
            voters: [...option.voters, randomStudent.id]
          };
        }
        // 如果是单选，需要移除该学生在其他选项的投票
        if (!currentPoll.allowMultiple) {
          return {
            ...option,
            votes: Math.max(0, option.votes - (option.voters.includes(randomStudent.id) ? 1 : 0)),
            voters: option.voters.filter(voterId => voterId !== randomStudent.id)
          };
        }
        return option;
      }),
      totalVotes: currentPoll.totalVotes + 1
    };

    setCurrentPoll(updatedPoll);
    setPolls(prev => prev.map(poll => 
      poll.id === currentPoll.id ? updatedPoll : poll
    ));

    toast({
      title: "✅ 投票成功",
      description: `${randomStudent.name} 已投票`
    });
  };

  // 批量随机投票
  const simulateBatchVoting = () => {
    if (!currentPoll || !currentPoll.isActive) return;

    const availableStudents = students.filter(student => {
      if (currentPoll.allowMultiple) {
        return true;
      } else {
        return !currentPoll.options.some(option => 
          option.voters.includes(student.id)
        );
      }
    });

    if (availableStudents.length === 0) {
      toast({
        title: "ℹ️ 无法批量投票",
        description: "所有学生都已投票",
        variant: "default"
      });
      return;
    }

    // 为每个可用学生随机分配选项
    const updatedOptions = [...currentPoll.options];
    let totalNewVotes = 0;

    availableStudents.forEach(student => {
      const randomOptionIndex = Math.floor(Math.random() * updatedOptions.length);
      const option = updatedOptions[randomOptionIndex];
      
      // 如果是单选，先移除该学生在其他选项的投票
      if (!currentPoll.allowMultiple) {
        updatedOptions.forEach(opt => {
          if (opt.voters.includes(student.id)) {
            opt.votes = Math.max(0, opt.votes - 1);
            opt.voters = opt.voters.filter(voterId => voterId !== student.id);
          }
        });
      }
      
      // 添加新投票
      option.votes += 1;
      option.voters.push(student.id);
      totalNewVotes += 1;
    });

    const updatedPoll = {
      ...currentPoll,
      options: updatedOptions,
      totalVotes: currentPoll.totalVotes + totalNewVotes
    };

    setCurrentPoll(updatedPoll);
    setPolls(prev => prev.map(poll => 
      poll.id === currentPoll.id ? updatedPoll : poll
    ));

    toast({
      title: "✅ 批量投票完成",
      description: `${availableStudents.length} 名学生已投票`
    });
  };

  // 删除投票
  const deletePoll = (pollId: string) => {
    setPolls(prev => prev.filter(poll => poll.id !== pollId));
    if (currentPoll && currentPoll.id === pollId) {
      setCurrentPoll(null);
      setShowResults(false);
    }
    
    toast({
      title: "🗑️ 投票已删除",
      description: "投票记录已删除"
    });
  };

  // 导出投票结果
  const exportResults = (poll: Poll) => {
    const data = {
      question: poll.question,
      description: poll.description,
      options: poll.options.map(option => ({
        text: option.text,
        votes: option.votes,
        percentage: poll.totalVotes > 0 ? ((option.votes / poll.totalVotes) * 100).toFixed(1) : '0'
      })),
      totalVotes: poll.totalVotes,
      isAnonymous: poll.isAnonymous,
      allowMultiple: poll.allowMultiple,
      createdAt: poll.createdAt.toISOString(),
      endedAt: poll.endedAt?.toISOString(),
      voterDetails: poll.isAnonymous ? null : poll.options.map(option => ({
        optionText: option.text,
        voters: option.voters.map(voterId => {
          const student = students.find(s => s.id === voterId);
          return student ? student.name : '未知学生';
        })
      }))
    };

    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `投票结果-${poll.question.slice(0, 20)}-${new Date().toLocaleDateString().replace(/\//g, '-')}.json`;
    link.click();
    URL.revokeObjectURL(url);

    toast({
      title: "📥 导出成功",
      description: "投票结果已导出"
    });
  };

  // 添加选项
  const addOption = () => {
    setNewPoll(prev => ({
      ...prev,
      options: [...prev.options, '']
    }));
  };

  // 移除选项
  const removeOption = (index: number) => {
    if (newPoll.options.length <= 2) return;
    setNewPoll(prev => ({
      ...prev,
      options: prev.options.filter((_, i) => i !== index)
    }));
  };

  // 更新选项文本
  const updateOption = (index: number, text: string) => {
    setNewPoll(prev => ({
      ...prev,
      options: prev.options.map((option, i) => i === index ? text : option)
    }));
  };

  return (
    <div className="space-y-6">
      {/* 头部控制区 */}
      <Card className="bg-gradient-to-r from-blue-50 to-indigo-50 border-blue-200">
        <CardHeader>
          <CardTitle className="flex items-center gap-3">
            <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-xl flex items-center justify-center">
              <Vote className="w-5 h-5 text-white" />
            </div>
            <div>
              <h2 className="text-2xl font-bold text-gray-900">课堂投票器</h2>
              <p className="text-sm text-gray-600 mt-1">实时互动投票，了解学生想法</p>
            </div>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap items-center gap-4">
            <Button 
              onClick={() => setIsCreating(true)}
              className="bg-gradient-to-r from-blue-500 to-indigo-600 hover:from-blue-600 hover:to-indigo-700"
            >
              <Plus className="w-4 h-4 mr-2" />
              创建投票
            </Button>
            
            {currentPoll && currentPoll.isActive && (
              <>
                <Button variant="outline" onClick={() => simulateBatchVoting()}>
                  <Shuffle className="w-4 h-4 mr-2" />
                  模拟批量投票
                </Button>
                <Button variant="outline" onClick={() => endPoll(currentPoll.id)}>
                  <Square className="w-4 h-4 mr-2" />
                  结束投票
                </Button>
              </>
            )}

            <div className="ml-auto flex items-center gap-2">
              <Badge variant="secondary">
                学生总数: {students.length} 人
              </Badge>
              {currentPoll && (
                <Badge variant={currentPoll.isActive ? "default" : "secondary"}>
                  {currentPoll.isActive ? "投票进行中" : "投票已结束"}
                </Badge>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* 创建投票表单 */}
      {isCreating && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Plus className="w-5 h-5 text-green-600" />
              创建新投票
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <Label>投票问题 *</Label>
              <Input
                value={newPoll.question}
                onChange={(e) => setNewPoll(prev => ({ ...prev, question: e.target.value }))}
                placeholder="输入投票问题..."
                className="mt-2"
              />
            </div>
            
            <div>
              <Label>问题描述（可选）</Label>
              <Textarea
                value={newPoll.description}
                onChange={(e) => setNewPoll(prev => ({ ...prev, description: e.target.value }))}
                placeholder="添加问题的详细描述..."
                className="mt-2"
                rows={3}
              />
            </div>

            <div>
              <Label>投票选项</Label>
              <div className="space-y-2 mt-2">
                {newPoll.options.map((option, index) => (
                  <div key={index} className="flex gap-2">
                    <Input
                      value={option}
                      onChange={(e) => updateOption(index, e.target.value)}
                      placeholder={`选项 ${index + 1}`}
                    />
                    {newPoll.options.length > 2 && (
                      <Button
                        variant="outline"
                        size="icon"
                        onClick={() => removeOption(index)}
                      >
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    )}
                  </div>
                ))}
                <Button variant="outline" onClick={addOption} className="w-full">
                  <Plus className="w-4 h-4 mr-2" />
                  添加选项
                </Button>
              </div>
            </div>

            <div className="flex flex-wrap gap-4">
              <label className="flex items-center gap-2">
                <input
                  type="checkbox"
                  checked={newPoll.isAnonymous}
                  onChange={(e) => setNewPoll(prev => ({ ...prev, isAnonymous: e.target.checked }))}
                  className="rounded"
                />
                <span className="text-sm">匿名投票</span>
              </label>
              
              <label className="flex items-center gap-2">
                <input
                  type="checkbox"
                  checked={newPoll.allowMultiple}
                  onChange={(e) => setNewPoll(prev => ({ ...prev, allowMultiple: e.target.checked }))}
                  className="rounded"
                />
                <span className="text-sm">允许多选</span>
              </label>
            </div>

            <div className="flex gap-3 pt-4">
              <Button onClick={createPoll} className="bg-green-600 hover:bg-green-700">
                <CheckCircle className="w-4 h-4 mr-2" />
                创建投票
              </Button>
              <Button variant="outline" onClick={() => setIsCreating(false)}>
                <XCircle className="w-4 h-4 mr-2" />
                取消
              </Button>
            </div>
          </CardContent>
        </Card>
      )}

      {/* 当前活跃投票 */}
      {currentPoll && currentPoll.isActive && (
        <Card className="border-green-200 bg-green-50">
          <CardHeader>
            <CardTitle className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <div className="w-3 h-3 bg-green-500 rounded-full animate-pulse" />
                <span>正在进行的投票</span>
              </div>
              <Badge variant="default" className="bg-green-600">
                <Clock className="w-3 h-3 mr-1" />
                进行中
              </Badge>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div>
                <h3 className="text-xl font-bold text-gray-900 mb-2">{currentPoll.question}</h3>
                {currentPoll.description && (
                  <p className="text-gray-600 mb-4">{currentPoll.description}</p>
                )}
              </div>

              <div className="grid gap-3">
                {currentPoll.options.map((option) => {
                  const percentage = currentPoll.totalVotes > 0 
                    ? (option.votes / currentPoll.totalVotes) * 100 
                    : 0;
                  
                  return (
                    <div key={option.id} className="space-y-2">
                      <div className="flex items-center justify-between">
                        <Button
                          variant="outline"
                          onClick={() => simulateVote(option.id)}
                          className="flex-1 justify-start hover:bg-blue-50"
                        >
                          <span className="flex-1 text-left">{option.text}</span>
                          <Badge variant="secondary">
                            {option.votes} 票 ({percentage.toFixed(1)}%)
                          </Badge>
                        </Button>
                      </div>
                      <Progress value={percentage} className="h-2" />
                    </div>
                  );
                })}
              </div>

              <div className="flex items-center justify-between pt-4 border-t">
                <div className="flex items-center gap-4 text-sm text-gray-600">
                  <span>总投票数: {currentPoll.totalVotes}</span>
                  <span>参与率: {students.length > 0 ? ((currentPoll.totalVotes / students.length) * 100).toFixed(1) : 0}%</span>
                  {currentPoll.isAnonymous && <Badge variant="outline">匿名</Badge>}
                  {currentPoll.allowMultiple && <Badge variant="outline">多选</Badge>}
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setShowResults(!showResults)}
                >
                  {showResults ? <EyeOff className="w-4 h-4 mr-1" /> : <Eye className="w-4 h-4 mr-1" />}
                  {showResults ? "隐藏" : "显示"}详情
                </Button>
              </div>

              {/* 投票详情 */}
              {showResults && !currentPoll.isAnonymous && (
                <div className="mt-4 p-4 bg-white rounded-lg border">
                  <h4 className="font-semibold mb-3">投票详情</h4>
                  <div className="space-y-3">
                    {currentPoll.options.map((option) => (
                      <div key={option.id}>
                        <div className="font-medium text-sm mb-1">{option.text}</div>
                        <div className="flex flex-wrap gap-1">
                          {option.voters.map(voterId => {
                            const student = students.find(s => s.id === voterId);
                            return student ? (
                              <Badge key={voterId} variant="outline" className="text-xs">
                                {student.name}
                              </Badge>
                            ) : null;
                          })}
                          {option.voters.length === 0 && (
                            <span className="text-xs text-gray-400">暂无投票</span>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      )}

      {/* 投票历史 */}
      {polls.length > 0 && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <BarChart3 className="w-5 h-5 text-purple-600" />
              投票历史
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {polls.map((poll) => (
                <div key={poll.id} className={`p-4 rounded-lg border-2 ${
                  poll.isActive 
                    ? 'border-green-200 bg-green-50' 
                    : 'border-gray-200 bg-gray-50'
                }`}>
                  <div className="flex items-start justify-between mb-3">
                    <div className="flex-1">
                      <h3 className="font-semibold text-gray-900">{poll.question}</h3>
                      {poll.description && (
                        <p className="text-sm text-gray-600 mt-1">{poll.description}</p>
                      )}
                      <div className="flex items-center gap-2 mt-2">
                        <Badge variant="outline" className="text-xs">
                          {poll.totalVotes} 票
                        </Badge>
                        <Badge variant="outline" className="text-xs">
                          {poll.createdAt.toLocaleDateString()} {poll.createdAt.toLocaleTimeString().slice(0, 5)}
                        </Badge>
                        {poll.isAnonymous && <Badge variant="outline" className="text-xs">匿名</Badge>}
                        {poll.allowMultiple && <Badge variant="outline" className="text-xs">多选</Badge>}
                      </div>
                    </div>
                    
                    <div className="flex items-center gap-2">
                      {!poll.isActive && (
                        <>
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => startPoll(poll.id)}
                          >
                            <Play className="w-4 h-4 mr-1" />
                            重新开始
                          </Button>
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => exportResults(poll)}
                          >
                            <Download className="w-4 h-4 mr-1" />
                            导出
                          </Button>
                        </>
                      )}
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => deletePoll(poll.id)}
                      >
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>

                  {/* 选项结果预览 */}
                  <div className="space-y-2">
                    {poll.options.map((option) => {
                      const percentage = poll.totalVotes > 0 
                        ? (option.votes / poll.totalVotes) * 100 
                        : 0;
                      
                      return (
                        <div key={option.id} className="flex items-center gap-3">
                          <span className="text-sm flex-1">{option.text}</span>
                          <div className="flex items-center gap-2 min-w-0">
                            <Progress value={percentage} className="w-20 h-2" />
                            <span className="text-xs text-gray-600 min-w-fit">
                              {option.votes} ({percentage.toFixed(1)}%)
                            </span>
                          </div>
                        </div>
                      );
                    })}
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {/* 空状态 */}
      {polls.length === 0 && !isCreating && (
        <Card className="text-center py-12">
          <CardContent>
            <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <Vote className="w-8 h-8 text-gray-400" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900 mb-2">还没有投票</h3>
            <p className="text-gray-600 mb-4">创建第一个投票，开始课堂互动吧！</p>
            <Button onClick={() => setIsCreating(true)}>
              <Plus className="w-4 h-4 mr-2" />
              创建投票
            </Button>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
